// This interface defines the metric space to be used in a SpaceWar game.
// This will normally be Euclidean, but one can imagine variations where
// the edges of the universe wrap around rather than bouncing.
//
// Note that strictly graphical tranformations are best handled in
// SpaceGraphics rather than here.

package SpaceWar;

import java.awt.*;
import java.awt.geom.*;

interface MetricSpace extends SpaceBackground {

	// Set the size of the space, presumably in response to a change
	// in frame size.

	public void setSize(int mnX, int mxX, int mnY, int mxY);

	// Check for collisions, given initial and final points for a
	// pair of objects, along with their radii.  Simple implementations
	// might simply check the pairs of points, and more complex
	// implementations might solve for closest approach assuming
	// linear motion.  Returns true if a collision was detected.

	boolean checkCollision(Point2D.Double p1,
					       Point2D.Double p2,
					       double pr,
					       Point2D.Double q1,
					       Point2D.Double q2,
					       double qr);

	// Compute the distance between a pair of objects.  Return the
	// distance, and also compute unit direction vectors from each
	// object to the other.  Note that the shortest distance may
	// be wrapped around the edge of the screen...
	//
	// If you don't want the unit vectors, pass null in for u1 and/or u2.

	double computeDistance(Point2D.Double p1,
					       Point2D.Double p2,
					       Point2D.Double u1,
					       Point2D.Double u2);

	// Scale a position vector.  Note that this is simple linear
	// scaling in a Euclidean metric space, and (for example) modular
	// scaling in a Riemannian metric space.
	
	void scalePosition(Point2D.Double p, double scale);
	
	// Add a pair of position vectors, placing the result in the
	// first vector.
	
	void addPosition(Point2D.Double p1, Point2D.Double p2);

	// Scale a velocity vector.  Note that this is simple linear
	// scaling in a Euclidean metric space, and (for example) Lorenz
	// transformations in the real universe.

	void scaleVelocity(Point2D.Double p, double scale);
	
	// Add a pair of velocity vectors, placing the result in the
	// first vector.
	
	void addVelocity(Point2D.Double p1, Point2D.Double p2);

	// Scale an acceleration vector.  Note that this is simple linear
	// scaling in a Euclidean metric space.  Alternate transformations
	// for alternate metric spaces are left as an exercise.  ;-)

	void scaleAcceleration(Point2D.Double p, double scale);
	
	// Add a pair of acceleration vectors, placing the result in the
	// first vector.
	
	void addAcceleration(Point2D.Double p1, Point2D.Double p2);

	// Update a position, given the object's acceleration, velocity,
	// initial position, and radius.  Returns true if the motion
	// was continuous, and false if it was discontinuous (e.g., it
	// bounced or wrapped).  Knowledge of discontinuity is important
	// for display methods that simulate speed blur.

	boolean updatePosition(Point2D.Double acceleration,
					       Point2D.Double velocity,
					       Point2D.Double position,
					       double radius);
	
	// Convert a fraction in the range [0,1] to a corresponding integer
	// denoting the corresponding display x-coordinate.
	
	int fractionToX(double xFraction);
	
	// Convert a fraction in the range [0,1] to a corresponding integer
	// denoting the corresponding display y-coordinate.
	
	int fractionToY(double xFraction);
			  
	// Paint any features that are a function of the metric space.
	// This will typically be a boundary marker.
	
	void paint(SpaceGraphics g);
}
