#include <stdio.h>
#include <stdlib.h>
#include <string.h>

int SUBORDER = 3;	/* Puzzle subdivisions (dump/print). */
int ORDER = 9;		/* Number of bits to track. */
int nrows = 9;		/* Number of rows, read/dump/print/index-conversion. */
int ncols = 9;		/* Number of cols, significant for read/dump/print. */
int nentries = 81;	/* Number of entries: nrows * ncols. */

int html = 0;
int verbosity = 0;
int do_twins = 1;	/* Smarter, but simple backtracking can be faster! */
int do_triplets = 0;	/* Ditto. */
int do_backtracking = 1;/* If you want, you can disable backtracking. */
int do_findmultiple = 0;/* Check for multiple solutions. */


/* #define  */

typedef int puzzle_entry_t;
typedef int (find_func_t)(puzzle_entry_t **h,
			  int *hidx, puzzle_entry_t *maskret);
typedef void (gather_func_t)(puzzle_entry_t *p, int li, puzzle_entry_t **h);

void compute_order(int linelen);

extern gather_func_t *gather_funcs[];
extern char *charmap;

/* Verbosity control. */

#define VERBOSE_PRINT_SUMMARY() 	(verbosity >= 1)
#define VERBOSE_PRINT_GUESS()		(verbosity >= 2)
#define VERBOSE_PRINT_STEPS()		(verbosity >= 3)
#define VERBOSE_PRINT_STEP_CHANGES()	(verbosity >= 4)
#define VERBOSE_DUMP_STEPS()		(verbosity >= 5)

/* Marking definitions to limit redundant application of techniques. */

#define MARK_SINGLETON			0x1
#define MARK_TWIN			0x2
#define MARK_TRIPLET			0x4

/* Control-mask definitions. */

#define CTLMASK_DO_TWINS		0x1
#define CTLMASK_DO_TRIPLETS		0x2
#define CTLMASK_DO_BACKTRACK		0x4

/* Return codes. */

#define PUZZLE_INCONSISTENT		-1
#define PUZZLE_UNSOLVED			0
#define PUZZLE_SOLVED			1
#define IS_UNSOLVED(ret) ((ret) != PUZZLE_SOLVED)
#define IS_SOLVED(ret)   ((ret) == PUZZLE_SOLVED)

/*
 * Return a pointer to the (i,j)th puzzle entry of a puzzle.
 * Row is selected by "i", column by "j".
 */

puzzle_entry_t *puzzle_entryp(puzzle_entry_t *p, int i, int j)
{
	return &p[i * ncols + j];
}

/*
 * Return the value of the (i,j)th puzzle entry of a puzzle.
 * Row is selected by "i", column by "j".
 */

puzzle_entry_t puzzle_entry(puzzle_entry_t *p, int i, int j)
{
	return *puzzle_entryp(p, i, j);
}

/*
 * Given a linear index into the puzzle, compute the row/column indices.
 */

void linear_index2rowcol(int li, int *i, int *j)
{
	if ((li < 0) || (li >= nentries)) {
		fprintf(stderr, "Bad linear index %d\n", li);
		abort();
	}
	*i = li / ncols;
	*j = li % ncols;
}

/*
 * Return a mask with all bits corresponding to the current ORDER.
 */

static inline puzzle_entry_t all_possible_bits()
{
	return (1 << ORDER) - 1;
}

/*
 * Return the number of bits set in "mask".
 */

int mask2count(puzzle_entry_t mask)
{
	int bit;
	static char *bits = NULL;
	int count;
	int i;

	if (bits == NULL) {

		/* First call, initialize. */

		bits = (char *)malloc(all_possible_bits() + 1);
		for (i = 0; i <= all_possible_bits(); i++) {
			count = 0;
			for (bit = 0; bit < ORDER; bit++) {
				if (i & (1 << bit)) {
					count++;
				}
			}
			bits[i] = count;
		}
	}
	if ((mask < 0) || (mask > all_possible_bits())) {
		fprintf(stderr, "Mask out of range!\n");
		abort();
	}
	return bits[mask];
}

/*
 * If "mask" has a single bit set, return the numeral corresponding to
 * that bit.  The least significant bit corresponds to '1', the tenth
 * least-significant bit to 'A', and so on.  If "mask" has multiple
 * bits set, return a blank.  If "mask" is out of range, use "?".
 */

char bit2numeral(puzzle_entry_t mask)
{
	static char *rep = NULL;

	if (rep == NULL) {
		char c;
		int i;

		/* First invocation, initialize. */

		rep = (char *)malloc(all_possible_bits() + 1);
		for (i = 0; i <= all_possible_bits(); i++) {
			rep[i] = ' ';
		}
		for (i = 0; (1 << i) <= all_possible_bits(); i++) {
			rep[1 << i] = charmap[i];
		}
	}
	if ((mask < 0) || (mask > all_possible_bits())) {
		return '?';
	}
	return rep[mask];
}

/*
 * Make a copy of the specified puzzle and return a pointer to it.
 */

puzzle_entry_t *copy_puzzle(puzzle_entry_t *puzzle)
{
	int *p1 = (int *)malloc(sizeof(p1[0]) * nentries * 2);
	int i1;

	if (p1 == NULL) {
		fprintf(stderr, "Out of memory.\n");
		exit(-1);
	}
	for (i1 = 0; i1 < nentries * 2; i1++) {
		p1[i1] = puzzle[i1];
	}
	return (p1);
}

/*
 * Dump the puzzle in hex regardless of the verbosity setting.
 */

void dump_puzzle_force(puzzle_entry_t *p)
{
	int i;
	int j;

	for (i = 0; i < nrows; i++) {
		for (j = 0; j < ncols; j++) {
			printf("%04x ", puzzle_entry(p, i, j));
			if (((j + 1) % SUBORDER == 0) && (j + 1 < ncols)) {
				printf("| ");
			}
		}
		printf("\n");
		if (((i + 1) % SUBORDER == 0) && (i + 1 < nrows)) {
			for (j = 0; j < ncols; j++) {
				printf("-----");
				if (((j + 1) % SUBORDER == 0) &&
				    (j + 1 < ncols)) {
					printf("+-");
				}
			}
			printf("\n");
		}
	}
}

/*
 * Dump the puzzle in hex if the verbosity setting is sufficient.
 */

void dump_puzzle(puzzle_entry_t *p)
{
	if (VERBOSE_DUMP_STEPS()) {
		dump_puzzle_force(p);
	}
}

/*
 * Read a square Sudoku puzzle and return a pointer to it.
 * Die on format error.
 */

puzzle_entry_t *read_puzzle(void)
{
	char c;
	int i;
	int j;
	int k;
	char inputline[4096];
	int linelen;
	int lineno;
	puzzle_entry_t *p = NULL;
	puzzle_entry_t *pp = NULL;

	lineno = 0;
	i = 0;
	for (;;) {
		if (fgets(inputline, sizeof(inputline), stdin) == NULL) {
			fprintf(stderr, "Only %d lines, need %d\n", i, nrows);
			exit(-1);
		}
		lineno += 1;
		if ((inputline[0] == '#') || (inputline[0] == '-')) {
			continue;
		}
		if (inputline[0] == '*') {
			char *cm;

			if (pp != NULL) {
				fprintf(stderr,
				"line %d: charmap line must precede input\n",
				lineno);
				exit(-1);
			}
			cm = (char *)malloc(strlen(inputline));
			if (cm == NULL) {
				fprintf(stderr, "line %d: out of memory.\n",
					lineno);
				exit(-1);
			}
			strcpy(cm, &inputline[1]);
			charmap = cm;
			charmap[strlen(charmap) - 1] = '\0';
			continue;
		}
		linelen = strlen(inputline);
		if (pp == NULL) {
			compute_order(linelen);
			if (strlen(charmap) < ORDER) {
				fprintf(stderr,
					"line %d: too small, need %d symbols.\n",
					lineno, ORDER);
				exit(-1);
			}
			p = (int *)malloc(sizeof(p[0]) * nrows * ncols * 2);
			bzero(&p[nrows * ncols], sizeof(p[0]) * nrows * ncols);
			pp = p;
		}
		if ((linelen != ncols * 2) ||
		    (inputline[ncols * 2 - 1] != '\n')) {
			fprintf(stderr, "line %d: malformed.\n", lineno);
			exit(-1);
		}
		for (j = 0; j < ncols; j++) {
			c = inputline[2 * j];
			*pp = 0;
			for (k = 0; charmap[k] != '\0'; k++) {
				if (c == charmap[k]) {
					if (k < ORDER) {
						*pp = 1 << k;
					}
					break;
				}
			}
			if (c == ' ') {
				*pp = all_possible_bits();
			}
			if ((*pp & all_possible_bits()) == 0) {
				fprintf(stderr,
					"line %d: symbol out of range '%c'\n",
					lineno, c);
				exit(-1);
			}
			pp++;
			c = inputline[2 * j + 1];
			if ((c != ' ') && (c != '|') && (c != '\n')) {
				fprintf(stderr,
				        "malformed input line %d\n", lineno);
				exit(-1);
			}
		}
		i++;
		if (i >= nrows) {
			break;
		}
	}
	dump_puzzle(p);
	return p;
}

/*
 * Print a Sudoku puzzle.
 */

void print_puzzle(puzzle_entry_t *p)
{
	int i;
	int j;
	puzzle_entry_t mask;

	for (i = 0; i < nrows; i++) {
		for (j = 0; j < ncols; j++) {
			mask = puzzle_entry(p, i, j);
			if (mask2count(mask) != 1) {
				printf("  ");
			} else {
				printf("%c ", bit2numeral(mask));
			}
			if (((j + 1) % SUBORDER == 0) && (j + 1 < ncols)) {
				printf("| ");
			}
		}
		printf("\n");
		if (((i + 1) % SUBORDER == 0) && (i + 1 < nrows)) {
			for (j = 0; j < ncols; j++) {
				printf("--");
				if (((j + 1) % SUBORDER == 0) &&
				    (j + 1 < ncols)) {
					printf("+-");
				}
			}
			printf("\n");
		}
	}
}

/*
 * Print a Sudoku puzzle as HTML, showing possibilities for unsolved cells.
 */

void print_puzzle_html(puzzle_entry_t *p)
{
	int first;
	int i;
	int j;
	int k;
	puzzle_entry_t mask;

	printf("<TABLE BORDER=3>\n");
	for (i = 0; i < nrows; i++) {
		printf("\t<TR>\n");
		for (j = 0; j < ncols; j++) {
			printf("\t\t<TD>");
			mask = puzzle_entry(p, i, j);
			first = 1;
			for (k = 0; k < ORDER; k++) {
				if (mask & (1 << k)) {
					if (first) {
						printf("%c",
						       bit2numeral(1 << k));
						first = 0;
					} else {
						printf(",%c",
						       bit2numeral(1 << k));
					}
				}
			}
			printf("</TD>\n");
		}
		printf("\t</TR>\n");
	}
	printf("</TABLE>\n");
}

/*
 * Update the element to the new set of bits, returning the number of
 * bits removed.
 */

int update_bits(puzzle_entry_t *element, puzzle_entry_t newbits)
{
	int count;

	if (newbits & ~*element) {
		fprintf(stderr, "Setting bits!!!");
		abort();
	}
	if (*element == newbits) {
		return 0;
	}
	count = mask2count(*element ^ newbits);
	*element = newbits;
	return count;
}

/*
 * Update the mark corresponding to the specified element.
 */

void update_mark(puzzle_entry_t *element, int newmark)
{
	element[nentries] |= newmark;
}

/*
 * Test the mark corresponding to the specified element.
 */

int test_mark(puzzle_entry_t *element, int newmark)
{
	return element[nentries] & newmark;
}

/*
 * Remove the specified bits from the element, returning the number
 * actually removed.
 */

int remove_bits(puzzle_entry_t *element, puzzle_entry_t bitstoremove)
{
	return update_bits(element, *element & ~bitstoremove);
}

/*
 * Remove the specified bits from any of the specified equivalence class
 * that is not already equal to the bits to be removed.  Return the number
 * of bits removed.
 */

int remove_bits_from_class(puzzle_entry_t **group, puzzle_entry_t removebits)
{
	int count = 0;
	int i;
	int oldentry;

	for (i = 0; i < ORDER; i++) {
		if (*group[i] != removebits) {
			count += update_bits(group[i], *group[i] & ~removebits);
		}
	}
	return count;
}

/*
 * Locates cases where an element is the only one with a given bit
 * set.  Returns a mask indicating the bit, or zero if there is no
 * such element.  Returns a mask with multiple bits set for inconsistent
 * puzzles.  The caller can print the error message.
 */

puzzle_entry_t find_inverse_singleton(puzzle_entry_t **h,
				      int *hidx, puzzle_entry_t *maskret)
{
	int bitcount;
	int count = 0;
	int i1;
	int j1;
	puzzle_entry_t mask;
	puzzle_entry_t singleton;

	for (i1 = 0; i1 < ORDER; i1++) {
		if (mask2count(*h[i1]) == 1) {
			continue;
		}
		mask = 0;
		for (j1 = 0; j1 < ORDER; j1++) {
			if (i1 != j1) {
				mask |= *h[j1];
			}
		}
		singleton = *h[i1] & ~mask;
		bitcount = mask2count(singleton);
		if (bitcount == 1) {
			count += update_bits(h[i1], singleton);
		} else if (bitcount != 0) {
			*hidx = i1;
			*maskret = mask;
			return count;
		}
	}
	*maskret = 0;
	return count;
}

/*
 * Locates cases where a pair of elements are the only ones with a given
 * pair of bits set.  Clears all other bits from the pair, and clears
 * the pair of bits from all others in the equivalence class.  Returns
 * the number of changes.
 * The caller can print the error message.
 */

int find_twins(puzzle_entry_t **h, int *hidx, puzzle_entry_t *maskret)
{
	int bitcount;
	int count = 0;
	int h1;
	int i1;
	int j1;
	puzzle_entry_t mask;
	puzzle_entry_t mask_i;
	puzzle_entry_t mask_j;

	for (i1 = 0; i1 < ORDER; i1++) {
		if (test_mark(h[i1], MARK_TWIN) ||
		    (mask2count(*h[i1]) < 2)) {
			continue;
		}
		for (j1 = i1 + 1; j1 < ORDER; j1++) {
			if (test_mark(h[j1], MARK_TWIN) ||
			    (mask2count(*h[j1]) < 2)) {
				continue;
			}
			mask = 0;
			for (h1 = 0; h1 < ORDER; h1++) {
				if ((i1 != h1) &&
				    (j1 != h1)) {
					mask |= *h[h1];
				}
			}
			mask_i = *h[i1] & ~mask;
			mask_j = *h[j1] & ~mask;
			mask = mask_i | mask_j;
			bitcount = mask2count(mask);
			if (bitcount < 2) {
				continue;
			}
			if (bitcount > 2) {

				/*
				 * There are more than two bits that can
				 * only be set in this pair of elements --
				 * broken puzzle or need to backtrack.
				 */

				*hidx = i1;
				*maskret = mask;
				return -1;
			}
			count += update_bits(h[i1], mask_i);
			update_mark(h[i1], MARK_TWIN);
			count += update_bits(h[j1], mask_j);
			update_mark(h[j1], MARK_TWIN);
			for (h1 = 0; h1 < ORDER; h1++) {
				if ((i1 != h1) &&
				    (j1 != h1)) {
					count += remove_bits(h[h1], mask);
				}
			}
		}
	}
	*maskret = 0;
	return count;
}

/*
 * Locates cases where a triplets of elements are the only ones with a given
 * pair of bits set.  Clears all other bits from the triple, and clears
 * the triplets of bits from all others in the equivalence class.  Returns
 * the number of changes.
 * The caller can print the error message.
 */

int find_triplets(puzzle_entry_t **h, int *hidx, puzzle_entry_t *maskret)
{
	int bitcount;
	int count = 0;
	int h1;
	int i1;
	int j1;
	int k1;
	puzzle_entry_t mask;
	puzzle_entry_t mask_i;
	puzzle_entry_t mask_j;
	puzzle_entry_t mask_k;

	for (i1 = 0; i1 < ORDER; i1++) {
		if (test_mark(h[i1], MARK_TRIPLET) ||
		    (mask2count(*h[i1]) < 3)) {
			continue;
		}
		for (j1 = i1 + 1; j1 < ORDER; j1++) {
			if (test_mark(h[j1], MARK_TRIPLET) ||
			    (mask2count(*h[j1]) < 3)) {
				continue;
			}
			for (k1 = j1 + 1; k1 < ORDER; k1++) {
				if (test_mark(h[k1], MARK_TRIPLET) ||
				    (mask2count(*h[j1]) < 3)) {
					continue;
				}
				mask = 0;
				for (h1 = 0; h1 < ORDER; h1++) {
					if ((i1 != h1) &&
					    (j1 != h1) &&
					    (k1 != h1)) {
						mask |= *h[h1];
					}
				}
				mask_i = *h[i1] & ~mask;
				mask_j = *h[j1] & ~mask;
				mask_k = *h[k1] & ~mask;
				mask = mask_i | mask_j | mask_k;
				bitcount = mask2count(mask);
				if (bitcount < 3) {
					continue;
				}
				if (bitcount > 3) {

					/*
					 * There are more than two bits
					 * that can only be set in this
					 * pair of elements -- broken
					 * puzzle or need to backtrack.
					 */

					*hidx = i1;
					*maskret = mask;
					return -1;
				}
				count += update_bits(h[i1], mask_i);
				update_mark(h[i1], MARK_TRIPLET);
				count += update_bits(h[j1], mask_j);
				update_mark(h[j1], MARK_TRIPLET);
				count += update_bits(h[k1], mask_k);
				update_mark(h[k1], MARK_TRIPLET);
				for (h1 = 0; h1 < ORDER; h1++) {
					if ((i1 != h1) &&
					    (j1 != h1) &&
					    (k1 != h1)) {
						count += remove_bits(h[h1],
								     mask);
					}
				}
			}
		}
	}
	*maskret = 0;
	return count;
}

/*
 * Locates inconsistencies, where an equivalence class does not represent
 * all values.
 */

int find_inconsistencies(puzzle_entry_t **h, int *hidx, puzzle_entry_t *maskret)
{
	int i1;
	puzzle_entry_t mask = 0;

	for (i1 = 0; i1 < ORDER; i1++) {
		mask |= *h[i1];
	}
	*maskret = 0;
	*hidx = 0;
	return ORDER - mask2count(mask);
}

/*
 * Remove the bit in the singleton at li from all elements sharing
 * an equivalence class with li.  Return the number of bits removed.
 */

int apply_singleton(puzzle_entry_t *p, int li)
{
	int count = 0;
	int f;
	int **h;
	int singleton = p[li];

	h = (int **)malloc(sizeof(h[0]) * ORDER);
	for (f = 0; gather_funcs[f] != NULL; f++) {
		gather_funcs[f](p, li, h);
		count += remove_bits_from_class(h, singleton);
	}
	free(h);
	update_mark(&p[li], MARK_SINGLETON);

	return count;
}

/*
 * Determine if p(i,j), or anything in its equivalence class, meets
 * the criterion specified by f.
 */

int check_equivalence(puzzle_entry_t *p, int li, find_func_t ff)
{
	int count = 0;
	int f;
	int **h;
	int hidx;
	int i1;
	int j1;
	puzzle_entry_t mask;
	puzzle_entry_t singleton = p[li];

	h = (int **)malloc(sizeof(h[0]) * ORDER);
	for (f = 0; gather_funcs[f] != NULL; f++) {
		gather_funcs[f](p, li, h);
		count += ff(h, &hidx, &mask);
		if (mask != 0) {
			return -1;
		}
	}
	free(h);
	return count;
}

/*
 * Scan the puzzle for singletons, and apply any that are found.
 * Return the number of bits cleared.
 */

int apply_all_singletons(puzzle_entry_t *p)
{
	int count = 0;
	int li;

	for (li = 0; li < nentries; li++) {
		if ((mask2count(p[li]) == 1) &&
		    (test_mark(&p[li], MARK_SINGLETON) == 0)) {
			count += apply_singleton(p, li);
		}
	}
	return count;
}

/*
 * Checks each element for inverse singletons and applies them.
 * Returns the number of bits actually cleared, or -1 if there
 * was an inconsistency.
 */

int apply_all_equivalence(puzzle_entry_t *p, find_func_t f)
{
	int count = 0;
	int curcount;
	int li;

	for (li = 0; li < nentries; li++) {
		curcount = check_equivalence(p, li, f);
		if (curcount < 0) {
			return -1;
		}
		count += curcount;
	}
	return count;
}

/*
 * Returns zero if no inconsistencies found in the puzzle.
 * @@@ inefficient -- scans all elements rather than simply all
 * equivalence classes.  Not worried just now.  ;-)
 */

int is_inconsistent(puzzle_entry_t *puzzle)
{
	return apply_all_equivalence(puzzle, find_inconsistencies);
}

/*
 * Returns zero if the puzzle has been solved.
 */

int puzzle_done(puzzle_entry_t *puzzle)
{

	int count_unsolved = 0;
	int count_wrong;
	int i;

	count_wrong = is_inconsistent(puzzle);
	if (count_wrong < 0) {
		count_wrong = 1;
	}
	for (i = 0; i < nentries; i++) {
		if (mask2count(puzzle[i]) != 1) {
			count_unsolved++;
		}
	}
	if (VERBOSE_PRINT_SUMMARY()) {
		printf("Number of inconsistencies: %d\n", count_wrong);
		printf("Number of incompletes: %d\n", count_unsolved);
	}
	return count_unsolved + count_wrong == 0;
}

/*
 * Clear the j-th bit of the i-th element of a copy of the puzzle, then
 * attempt to solve the resulting puzzle.
 */

int do_one_backtrack(puzzle_entry_t *puzzle, int entry, int bitno, int ctlmask)
{
	int bitcount;
	int count;
	int i1;
	int *p1 = copy_puzzle(puzzle);
	int ret;
	int solve_puzzle(puzzle_entry_t *puzzle, int ctlmask);

	if (VERBOSE_PRINT_GUESS()) {
		printf("Guessing %d/%d (%x)\n", entry, bitno, p1[entry]);
	}
	bitcount = 0;
	for (i1 = 0; i1 < ORDER; i1++) {
		if ((1 << i1) & p1[entry]) {
			if (bitcount == bitno) {

				/* Found bitno-th bit in specified entry. */

				p1[entry] &= ~(1 << i1);
				ret = solve_puzzle(p1, ctlmask);
				if (IS_UNSOLVED(ret)) {
					if (VERBOSE_PRINT_GUESS()) {
						printf("Failed guess %d/%d\n",
						       entry, bitno);
					}
				} else {

					/* Solved, copy back solution.*/

					for (i1 = 0; i1 < nentries * 2; i1++) {
						puzzle[i1] = p1[i1];
					}
					if (VERBOSE_PRINT_GUESS()) {
						printf("Succeeded guess %d/%d\n",
						       entry, bitno);
					}
				}
				free(p1);
				return (ret);
			}
			bitcount++;
		}
	}

	/* There was no j-th bit in the entry-th element!!! */

	fprintf(stderr, "Backtrack failed\n");
	abort();
}

/*
 * Select bits to remove from puzzle, starting with the elements with
 * two bits set, then doing those with three bits set, and so on.
 * Note that we clear only one bit at a time -- if it is necessary to
 * clear multiple bits, we will do it in a recursive call to this
 * function.
 */

int do_backtrack(puzzle_entry_t *puzzle, int ctlmask)
{
	int i;
	int j;
	int nbits;

	for (nbits = 2; nbits <= ORDER; nbits++) {
		for (i = 0; i < nentries; i++) {
			if (mask2count(puzzle[i]) == nbits) {
				for (j = 0; j < nbits; j++) {
					if (do_one_backtrack(puzzle,
							     i, j,
							     ctlmask) > 0) {
						return PUZZLE_SOLVED;
					}
					if (is_inconsistent(puzzle)) {
						return PUZZLE_INCONSISTENT;
					}
				}
			}
		}
	}
	return -1;
}

/*
 * Repeatedly apply strategies to solve puzzle.
 */

int solve_puzzle(puzzle_entry_t *puzzle, int ctlmask)
{
	int count;
	int count1;
	int countret = 0;
	int i;

	do {
		count1 = 0;
		do {
			if (VERBOSE_PRINT_STEPS()) {
				printf("---- apply_all_inverse_singletons()\n");
			}
			count = apply_all_equivalence(puzzle,
						      find_inverse_singleton);
			if (count < 0) {
				return -1;
			}
			count1 += count;
			if (VERBOSE_PRINT_STEP_CHANGES()) {
				printf("changed: %d\n", count);
			}
			dump_puzzle(puzzle);
		} while (count > 0);
		do {
			if (VERBOSE_PRINT_STEPS()) {
				printf("---- apply_all_singletons()\n");
			}
			count1 += count = apply_all_singletons(puzzle);
			if (VERBOSE_PRINT_STEP_CHANGES()) {
				printf("changed: %d\n", count);
			}
			dump_puzzle(puzzle);
		} while (count > 0);
		if (ctlmask & CTLMASK_DO_TWINS) {
			do {
				if (VERBOSE_PRINT_STEPS()) {
					printf("---- apply_all_twins()\n");
				}
				count = apply_all_equivalence(puzzle,
							      find_twins);
				if (count < 0) {
					return -1;
				}
				count1 += count;
				if (VERBOSE_PRINT_STEP_CHANGES()) {
					printf("changed: %d\n", count);
				}
				dump_puzzle(puzzle);
			} while (count > 0);
		}
		if (ctlmask & CTLMASK_DO_TRIPLETS) {
			do {
				if (VERBOSE_PRINT_STEPS()) {
					printf("---- apply_all_triplets()\n");
				}
				count = apply_all_equivalence(puzzle,
							      find_triplets);
				if (count < 0) {
					return -1;
				}
				count1 += count;
				if (VERBOSE_PRINT_STEP_CHANGES()) {
					printf("changed: %d\n", count);
				}
				dump_puzzle(puzzle);
			} while (count > 0);
		}
		countret += count1;
	} while (count1 != 0);
	if (VERBOSE_PRINT_SUMMARY()) {
		printf("---- %d elimination steps\n", countret);
	}
	if (puzzle_done(puzzle)){
		return PUZZLE_SOLVED;
	}
	if (ctlmask & CTLMASK_DO_BACKTRACK) {
		return do_backtrack(puzzle, ctlmask);
	}
	return PUZZLE_UNSOLVED;
}

/*
 * Given an initial solution and a copy of the original puzzle, find
 * at least one alternative solution if there is one.  Currently
 * finds -any- solution -- a more sophisticated approach might check
 * for the possibility of symbol interchange.
 */

puzzle_entry_t *find_other_solution(puzzle_entry_t *puzzle,
				    puzzle_entry_t *puzzle_orig,
				    int ctlmask)
{
	int i;
	int j;
	puzzle_entry_t mask;
	puzzle_entry_t *p1;
	int r;

	r = solve_puzzle(puzzle_orig, ctlmask & ~CTLMASK_DO_BACKTRACK);
	if (IS_UNSOLVED(r)) {
		for (i = 0; i < nentries; i++) {
			mask = puzzle_orig[i] & ~puzzle[i];
			if (mask == 0) {
				continue;
			}
			/* potential multiple solution. */
			p1 = copy_puzzle(puzzle_orig);
			for (j = 0; j < ORDER; j++) {
				if ((mask & (1 << j)) == 0) {
					continue;
				}
				p1[i] = 1 << j;
				r = solve_puzzle(p1, (ctlmask &
						      ~CTLMASK_DO_BACKTRACK));
				if (IS_SOLVED(r)) {
					return p1;
				}
				free(p1);
				p1 = copy_puzzle(puzzle_orig);
			}
			free(p1);
		}
	}
	return NULL;
}

void parse_args(int argc, char *argv[])
{
	int i;

	for (i = 1; i < argc; i++) {
		if (strcmp(argv[i], "--html") == 0) {
			html = 1;
		} else if (strcmp(argv[i], "--verbose") == 0) {
			if ((i + 1 == argc) || (argv[i + 1][0] == '-')) {
				verbosity = 99;
			} else {
				verbosity = strtoul(argv[i + 1], NULL, 0);
				i++;
			}
		} else if (strcmp(argv[i], "--quiet") == 0) {
			verbosity = 0;
		} else if (strcmp(argv[i], "--notwins") == 0) {
			do_twins = 0;
		} else if (strcmp(argv[i], "--twins") == 0) {
			do_twins = 1;
		} else if (strcmp(argv[i], "--notriplets") == 0) {
			do_triplets = 0;
		} else if (strcmp(argv[i], "--triplets") == 0) {
			do_triplets = 1;
		} else if (strcmp(argv[i], "--backtrack") == 0) {
			do_backtracking = 1;
		} else if (strcmp(argv[i], "--nobacktrack") == 0) {
			do_backtracking = 0;
		} else if (strcmp(argv[i], "--findmultiple") == 0) {
			do_findmultiple = 1;
		} else {
			fprintf(stderr, "Usage: %s "
				"[--html] "
				"[--verbose [n]] [--quiet]\n\t\t"
				"[--[no]twins] [--[no]triplets] "
				"[--[no]backtrack] "
				"[--findmultiple]\n", argv[0]);
			exit(-1);
		}
	}
}

/*
 * TODO:
 *
 * Consider adding the number of singletons in an equivalence class
 * as one of the factors guiding backtrack/search.  The greater the
 * number of singletons, the greater the effect of an arbitrary choice.
 *
 * Create difficulty vector to track count changes.  Just double the
 * length of the puzzle vector, and use bits to indicate what operations
 * have been done, e.g., singleton, twin, triplets.
 *
 * Currently the operation-bits don't take into account that a single
 * entry might participate in (say) twin operations in multiple
 * equivalence classes...  Need multiple bits, and ability to shift
 * up by equivalence-class number...  Singletons and inverse singletons
 * are currently handled specially.  Of course, the current behavior
 * is itself a strange heuristic that assumes that a single cell
 * participating in multiple twin/triplet operations is so rare that
 * backtracking is cheaper...
 *
 * Rename ORDER and SUBORDER?
 */

int main(int argc, char *argv[])
{
	int ctlmask;
	puzzle_entry_t *puzzle;
	puzzle_entry_t *puzzle_orig = NULL;
	int r;

	parse_args(argc, argv);
	puzzle = read_puzzle();
	if (do_findmultiple) {
		puzzle_orig = copy_puzzle(puzzle);
	}
	ctlmask = (do_twins ? CTLMASK_DO_TWINS : 0) |
		  (do_triplets ? CTLMASK_DO_TRIPLETS : 0) |
		  (do_backtracking ? CTLMASK_DO_BACKTRACK : 0);
	r = solve_puzzle(puzzle, ctlmask);
	if (html) {
		print_puzzle_html(puzzle);
	} else {
		if (IS_UNSOLVED(r)) {
			dump_puzzle_force(puzzle);
		}
		print_puzzle(puzzle);
	}
	if (IS_SOLVED(r) && do_findmultiple && do_backtracking) {
		puzzle_entry_t *p1;

		p1 = find_other_solution(puzzle, puzzle_orig, ctlmask);
		if (p1 != NULL) {
			printf("Multiple solutions!\n");
			if (html) {
				print_puzzle_html(p1);
			} else {
				print_puzzle(p1);
			}
			free(p1);
		}
	}
	free(puzzle);
	if (do_findmultiple) {
		free(puzzle_orig);
	}
	exit(0);
}
